bl_info = {
    "name": "Three.js Exporter (Lite) + Options",
    "author": "F.Heiland",
    "version": (0, 9, 3),
    "blender": (4, 0, 0),
    "location": "Scene > Three.js Exporter (Lite) & File > Export",
    "description": "Export scene as a minimal three.js project (GLB + HTML/JS with import map). Options: HDRI, debug helpers, use active viewport camera. Respects Hide in Viewport. Supports start-hidden/keyboard toggle for collection 'hide'.",
    "category": "Import-Export",
}

import bpy, os, math
from bpy.props import StringProperty, BoolProperty
from bpy.types import Operator, Panel
from bpy_extras.io_utils import ExportHelper

# -------------------- FILE TEMPLATES --------------------

HTML_TEMPLATE = """<!DOCTYPE html>
<html lang="no">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1.0"/>
  <title>{title}</title>
  <style>
    html,body{{margin:0;height:100%;overflow:hidden;background:#0b0d12;font-family:Arial,sans-serif}}
    #app{{position:fixed;inset:0}}
    .controls{{position:absolute;top:16px;left:16px;z-index:10;display:flex;gap:8px}}
    button{{padding:8px 12px;background:#4CAF50;color:#fff;border:0;border-radius:6px;cursor:pointer;font-size:14px}}
    button:hover{{background:#45a049}}
    .hint{{position:absolute;top:16px;right:16px;color:#cbd5e1;font:12px/1.4 system-ui, sans-serif;opacity:.85}}
    .hint kbd{{background:#1f2937;border:1px solid #334155;border-bottom-width:2px;border-radius:4px;padding:2px 6px}}
  </style>

  <!-- Import map: pin 'three' og 'three/addons/' til samme versjon -->
  <script type="importmap">
  {{
    "imports": {{
      "three": "https://unpkg.com/three@0.161.0/build/three.module.js",
      "three/addons/": "https://unpkg.com/three@0.161.0/examples/jsm/"
    }}
  }}
  </script>
</head>
<body>
  <div id="app"></div>
  <div class="controls">
    <button id="btn-fit">Reset kamera</button>
    <button id="btn-anim">Pause/Play animasjon</button>
  </div>
  <div class="hint">Toggle collection <strong>hide</strong> med <kbd>H</kbd></div>

  <script type="module" src="./main.js"></script>
</body>
</html>
"""

README_TXT = """Three.js Export (Lite)
=======================
Filer:
- index.html : viewer m/ import map (pinner three@0.161.0)
- main.js    : ES-modul; valg for HDRI/IBL, debug helpers og aktivt kamera
- scene.glb  : eksportert fra Blender

Kjør lokalt:
  python3 -m http.server 8080
  → http://localhost:8080

Prod:
  Last opp mappa til Caddy/NGINX/Traefik/Netlify/etc.
  VR krever HTTPS + WebXR-støttet enhet.

Tips:
  - Color Management → View Transform = Standard.
  - Bake shader-effekter til teksturer.
  - GLTF export inkluderer kamera/lys/animasjon.
  - Eksporten følger viewport-synlighet (øye-ikonet i Outliner). Objekter skjult i viewport blir ikke med.
  - Collection 'hide': i viewer kan du toggle synligheten med tast <H>. Du kan også starte skjult via panelet.
"""

HDRI_README = """Legg HDR-filen din her (samme domene = unngår CORS)
----------------------------------------------------------
Standardnavn i prosjektmalen: venice_sunset_1k.hdr

Du kan bytte navn/fil i main.js om du ønsker en annen HDR.
"""

# main.js bygges fra en trygg template (ikke f-string) for å unngå krøll med { } i JS
MAIN_JS_TEMPLATE = """import * as THREE from 'three';
import { OrbitControls } from 'three/addons/controls/OrbitControls.js';
import { GLTFLoader } from 'three/addons/loaders/GLTFLoader.js';
import { DRACOLoader } from 'three/addons/loaders/DRACOLoader.js';
import { KTX2Loader } from 'three/addons/loaders/KTX2Loader.js';
import { VRButton } from 'three/addons/webxr/VRButton.js';
import { RGBELoader } from 'three/addons/loaders/RGBELoader.js';

const app = document.getElementById('app');
const clock = new THREE.Clock();
let USED_ACTIVE_CAMERA = false;

// Renderer
const renderer = new THREE.WebGLRenderer({ antialias: true, alpha: true });
renderer.setPixelRatio(Math.min(devicePixelRatio, 2));
renderer.setSize(innerWidth, innerHeight);
renderer.outputColorSpace = THREE.SRGBColorSpace;
renderer.toneMapping = THREE.ACESFilmicToneMapping;
renderer.xr.enabled = true;
app.appendChild(renderer.domElement);

// Scene + kamera
const scene = new THREE.Scene();
let camera = new THREE.PerspectiveCamera(60, innerWidth / innerHeight, 0.1, 1000);
camera.position.set(3, 1.6, 5);
scene.add(camera);

// Lys
scene.add(new THREE.AmbientLight(0xffffff, 0.6));
const dir = new THREE.DirectionalLight(0xffffff, 1.0);
dir.position.set(5, 10, 7);
scene.add(dir);

// Debug helpers (valgfritt)
__HELPERS__
// IBL / HDRI (valgfritt)
__HDRI__

const controls = new OrbitControls(camera, renderer.domElement);
controls.enableDamping = true;

// Loaders
const gltfLoader = new GLTFLoader();
const draco = new DRACOLoader();
draco.setDecoderPath('https://unpkg.com/three@0.161.0/examples/jsm/libs/draco/');
gltfLoader.setDRACOLoader(draco);

const ktx2 = new KTX2Loader();
ktx2.setTranscoderPath('https://unpkg.com/three@0.161.0/examples/jsm/libs/basis/');
ktx2.detectSupport(renderer);
gltfLoader.setKTX2Loader(ktx2);

// State
let mixer = null;
let activeAction = null;
let gltf = null;
let hideGroup = null;

// (valgfritt) bruk aktivt Blender-kamera nå
__ACTIVE_CAM__

// Last GLB (samme mappe)
load('scene.glb');

async function load(url) {
  try {
    gltf = await gltfLoader.loadAsync(url);

    // Finn 'hide' node (Empty / group med navn 'hide')
    hideGroup = gltf.scene.getObjectByName('hide') || gltf.scene.getObjectByProperty('name', 'hide');

    // Start hidden (valgfritt)
    __HIDE_START__

    // Kamera fra GLB hvis finnes (GLB-kamera prioriteres)
    let gltfCamera = null;
    gltf.scene.traverse((o) => { if (o.isCamera && !gltfCamera) gltfCamera = o; });
    scene.add(gltf.scene);

    if (gltfCamera) {
      camera = gltfCamera;
      if (!camera.parent) scene.add(camera);
      controls.object = camera;
    } else {
      if (!USED_ACTIVE_CAMERA) {
        fitToObject(gltf.scene, camera, controls);
      }
    }

    if (gltf.animations?.length) {
      mixer = new THREE.AnimationMixer(gltf.scene);
      activeAction = mixer.clipAction(gltf.animations[0]);
      activeAction.play();
    }
  } catch (e) {
    console.error('Failed to load GLB:', e);
  }
}

function fitToObject(object, cam, controls, offset = 1.15) {
  const box = new THREE.Box3().setFromObject(object);
  if (box.isEmpty()) return;

  const size = box.getSize(new THREE.Vector3());
  const center = box.getCenter(new THREE.Vector3());
  const maxDim = Math.max(size.x, size.y, size.z);
  const fov = cam.isPerspectiveCamera ? cam.fov * (Math.PI / 180) : 45 * (Math.PI / 180);
  let cameraZ = (maxDim / 2) / Math.tan(fov / 2) * offset;
  cameraZ = Math.max(cameraZ, 0.1);

  cam.position.set(center.x + cameraZ, center.y + cameraZ * 0.4, center.z + cameraZ);
  cam.near = Math.max(0.01, maxDim / 100);
  cam.far  = Math.max(100, maxDim * 100);
  cam.updateProjectionMatrix();

  controls.target.copy(center);
  controls.update();
}

// Tastatursnarvei: H for å toggle 'hide'
window.addEventListener('keydown', (e) => {
  if (e.key.toLowerCase() === 'h' && hideGroup) {
    hideGroup.visible = !hideGroup.visible;
    console.log(`'hide' group: ${hideGroup.visible ? 'visible' : 'hidden'}`);
  }
});

// UI
document.getElementById('btn-fit')?.addEventListener('click', () => {
  fitToObject(gltf ? gltf.scene : scene, camera, controls, 1.2);
});
document.getElementById('btn-anim')?.addEventListener('click', () => {
  if (activeAction) activeAction.paused = !activeAction.paused;
});
document.getElementById('btn-vr')?.addEventListener('click', () => {
  if (!document.getElementById('VRButton')) {
    document.body.appendChild(VRButton.createButton(renderer));
  }
});

// Resize + render loop
window.addEventListener('resize', () => {
  camera.aspect = innerWidth / innerHeight;
  camera.updateProjectionMatrix();
  renderer.setSize(innerWidth, innerHeight);
});
renderer.setAnimationLoop(() => {
  const dt = clock.getDelta();
  if (mixer) mixer.update(dt);
  controls.update();
  renderer.render(scene, camera);
});
"""

def _write(path, txt):
    os.makedirs(os.path.dirname(path), exist_ok=True)
    with open(path, "w", encoding="utf-8") as f:
        f.write(txt)

# -------------------- OPERATOR --------------------

class THREEJS_OT_export_project(Operator, ExportHelper):
    bl_idname = "export_scene.three_js_project"
    bl_label = "Export Three.js Project"
    bl_options = {'REGISTER', 'UNDO'}

    filename_ext = ""
    filter_glob: StringProperty(default="*", options={'HIDDEN'})

    def invoke(self, context, event):
        context.window_manager.fileselect_add(self)
        return {'RUNNING_MODAL'}

    def execute(self, context):
        export_dir = self.filepath
        if os.path.splitext(export_dir)[1]:
            export_dir = os.path.dirname(export_dir)
        os.makedirs(export_dir, exist_ok=True)

        glb_path = os.path.join(export_dir, "scene.glb")

        # Alltid-definerte telleverk
        exported_objs = []
        visible_count = 0

        # ---------- 1) Eksporter GLB: kun objekter som IKKE er skjult i viewport ----------
        try:
            if bpy.context.mode != 'OBJECT':
                bpy.ops.object.mode_set(mode='OBJECT')

            ctx = bpy.context
            scene = ctx.scene
            view_layer = ctx.view_layer

            # Husk selection/active
            prev_selection = [o for o in ctx.selected_objects]
            prev_active = view_layer.objects.active if hasattr(view_layer.objects, "active") else ctx.active_object

            # Velg kun objekter som er synlige i viewport
            bpy.ops.object.select_all(action='DESELECT')

            for o in scene.objects:
                is_visible = o.visible_get(view_layer=view_layer)
                if is_visible:
                    visible_count += 1
                    o.select_set(True)
                    exported_objs.append(o)

            if ctx.selected_objects:
                view_layer.objects.active = ctx.selected_objects[0]

            bpy.ops.export_scene.gltf(
                filepath=glb_path,
                export_format='GLB',
                use_selection=True,
                export_yup=True,
                export_apply=True,
                export_texcoords=True,
                export_normals=True,
                export_tangents=False,
                export_materials='EXPORT',
                export_image_format='AUTO',
                export_cameras=True,
                export_lights=True,
                export_animations=True,
                export_skins=True,
                export_morph=True
            )

        except Exception as e:
            self.report({'ERROR'}, f"glTF export failed: {e}")
            return {'CANCELLED'}
        finally:
            # gjenopprett selection/active
            try:
                bpy.ops.object.select_all(action='DESELECT')
                for o in prev_selection:
                    if o and o.name in bpy.context.scene.objects:
                        o.select_set(True)
                if prev_active and prev_active.name in bpy.context.scene.objects:
                    view_layer.objects.active = prev_active
            except Exception:
                pass

        # ---------- 2) Bygg main.js med valgte alternativer ----------
        include_hdri = bool(context.scene.threejs_include_hdri)
        add_helpers = bool(context.scene.threejs_add_helpers)
        use_active  = bool(context.scene.threejs_use_active_cam)
        start_hidden_hide = bool(context.scene.threejs_start_hidden_hide)

        # Helpers
        helpers_block = ""
        if add_helpers:
            helpers_block = (
                "scene.background = new THREE.Color(0x1a1f2b);\n"
                "scene.add(new THREE.GridHelper(10, 10));\n"
                "scene.add(new THREE.AxesHelper(1.5));\n"
            )

        # HDRI: bruk lokal filsti (samme origin) for å unngå CORS
        hdri_block = ""
        if include_hdri:
            local_hdr_dir = os.path.join(export_dir, "textures", "equirectangular")
            os.makedirs(local_hdr_dir, exist_ok=True)
            _write(os.path.join(local_hdr_dir, "READ_ME.txt"), HDRI_README)

            hdri_block = (
                "new RGBELoader()\n"
                "  .setPath('./textures/equirectangular/')\n"
                "  .load('venice_sunset_1k.hdr', (hdr) => {\n"
                "    hdr.mapping = THREE.EquirectangularReflectionMapping;\n"
                "    scene.environment = hdr;\n"
                "    // scene.background = hdr; // slå på hvis du vil se skybox\n"
                "  }, undefined, (err) => {\n"
                "    console.warn('HDRI load failed, continuing without IBL:', err);\n"
                "  });\n"
            )

        # Aktivt Blender-kamera
        active_cam_block = ""
        if use_active and context.scene.camera and context.scene.camera.type == 'CAMERA':
            cam_obj = context.scene.camera
            cam_data = cam_obj.data
            loc = cam_obj.matrix_world.translation
            rot = cam_obj.matrix_world.to_quaternion()
            near = getattr(cam_data, "clip_start", 0.1)
            far  = getattr(cam_data, "clip_end", 1000.0)
            if cam_data.type == 'PERSP':
                fov_rad = getattr(cam_data, "angle", math.radians(60.0))
                fov_deg = fov_rad * 180.0 / math.pi
            else:
                fov_deg = 60.0
            active_cam_block = (
                "// Init fra Blender aktivt kamera\n"
                f"camera.fov = {fov_deg:.6f};\n"
                f"camera.near = {near:.6f};\n"
                f"camera.far = {far:.6f};\n"
                f"camera.position.set({loc.x:.6f}, {loc.y:.6f}, {loc.z:.6f});\n"
                f"camera.quaternion.set({rot.x:.6f}, {rot.y:.6f}, {rot.z:.6f}, {rot.w:.6f});\n"
                "camera.updateProjectionMatrix();\n"
                "controls.object = camera;\n"
                "USED_ACTIVE_CAMERA = true;\n"
            )

        # Start-hidden for 'hide' i JS
        hide_start_block = "if (hideGroup) hideGroup.visible = false;" if start_hidden_hide else ""

        # Sett sammen main.js fra template
        main_js = (
            MAIN_JS_TEMPLATE
            .replace("__HELPERS__", helpers_block)
            .replace("__HDRI__", hdri_block)
            .replace("__ACTIVE_CAM__", active_cam_block)
            .replace("__HIDE_START__", hide_start_block)
        )

        # ---------- 3) Skriv filer ----------
        try:
            title = bpy.path.display_name_from_filepath(bpy.data.filepath) or "Three.js Viewer"
            _write(os.path.join(export_dir, "index.html"), HTML_TEMPLATE.format(title=title))
            _write(os.path.join(export_dir, "main.js"), main_js)
            _write(os.path.join(export_dir, "README.txt"), README_TXT)
        except Exception as e:
            self.report({'ERROR'}, f"Failed to write template files: {e}")
            return {'CANCELLED'}

        # ---------- 4) Statusmelding ----------
        total = len(bpy.context.scene.objects)
        hidden_in_viewport = total - visible_count
        exported_count = len(exported_objs)

        self.report(
            {'INFO'},
            f"Exported three.js project to: {export_dir} "
            f"({exported_count} objects exported, {hidden_in_viewport} skipped as hidden in viewport)"
        )

        return {'FINISHED'}

# -------------------- UI PANEL --------------------

class THREEJS_PT_export_panel(Panel):
    bl_label = "Three.js Exporter (Lite)"
    bl_space_type = 'PROPERTIES'
    bl_region_type = 'WINDOW'
    bl_context = "scene"

    def draw(self, context):
        col = self.layout.column(align=True)
        col.label(text="Export scene as a static three.js project")
        col.prop(context.scene, "threejs_include_hdri", toggle=True)
        col.prop(context.scene, "threejs_add_helpers", toggle=True)
        col.prop(context.scene, "threejs_use_active_cam", toggle=True)
        col.separator()
        col.label(text="Collection 'hide' options:")
        col.prop(context.scene, "threejs_start_hidden_hide", toggle=True, text='Start hidden: collection "hide" (H to toggle)')
        col.separator()
        col.operator(THREEJS_OT_export_project.bl_idname, icon='EXPORT')

def menu_func(self, context):
    self.layout.operator(THREEJS_OT_export_project.bl_idname, text="Three.js Project (.glb + HTML/JS)")

# -------------------- REGISTER / UNREGISTER --------------------

def register():
    bpy.utils.register_class(THREEJS_OT_export_project)
    bpy.utils.register_class(THREEJS_PT_export_panel)
    bpy.types.TOPBAR_MT_file_export.append(menu_func)

    bpy.types.Scene.threejs_include_hdri = BoolProperty(
        name="Include HDRI (IBL)",
        description="Legg til HDRI miljølys (forventer HDR i ./textures/equirectangular/)",
        default=True,
    )
    bpy.types.Scene.threejs_add_helpers = BoolProperty(
        name="Add debug helpers (grid/axes)",
        description="Vis GridHelper og AxesHelper",
        default=False,
    )
    bpy.types.Scene.threejs_use_active_cam = BoolProperty(
        name="Use active viewport camera",
        description="Bruk aktivt Blender-kamera som startkamera hvis GLB ikke har kamera",
        default=True,
    )
    bpy.types.Scene.threejs_start_hidden_hide = BoolProperty(
        name='Start hidden: collection "hide"',
        description='Setter synligheten til gruppen "hide" til skjult i viewer, og lar deg toggle med H',
        default=True,
    )

def unregister():
    bpy.types.TOPBAR_MT_file_export.remove(menu_func)
    del bpy.types.Scene.threejs_include_hdri
    del bpy.types.Scene.threejs_add_helpers
    del bpy.types.Scene.threejs_use_active_cam
    del bpy.types.Scene.threejs_start_hidden_hide
    bpy.utils.unregister_class(THREEJS_PT_export_panel)
    bpy.utils.unregister_class(THREEJS_OT_export_project)

if __name__ == "__main__":
    register()
